/** @file   settings.h
 * @brief   Declaration of the Settings - class.
 * @version $Revision: 1.3 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_SETTINGS_H
#define H_WWW_SETTINGS_H

#include <vector>
#include <fstream>
#include <allegro.h>
#include "utils.h"
#include "www_exceptions.h"
#include "filelist.h"


namespace WeWantWar {

/** @class  Settings
 * @brief   Stores and manages the general settings of the game.
 * @author  Tomi Lamminsaari
 *
 * During the startup of the game, the <code> init() </code> - method
 * should be called. It loads the general settings from the configuration
 * file. These settings includes for example the moving velocities of the
 * objects and weapon firepowers.
 *
 */
class Settings
{
public:

  ///
  /// Static members and methods
  /// ==========================
  
  /** Name of the file where the settings from Settings - menu will be saved */
  static const std::string SAVE_SETTINGS_TO_FILE;
  
  
  /** ID for Controlmode 0. Mouse turns and shoots and moving is relative
   * to the player's angle.
   */
  static const int CONTROLMODE1 = 0;
  /** ID for Controlmode 1. Mouse aims and moving is absolute, so W-key
   * always moves the player up.
   */
  static const int CONTROLMODE2 = 1;
  
  
  /** The current control mode the player uses. */
  static int controlMode;
  
  /** Scancode for key that moves player forward or up */
  static int key_up;
  /** Scancode for key that moves player backward or down */
  static int key_down;
  /** Scancode for key that strafes player left or moves to the left */
  static int key_left;
  /** Scancode for key that strafes player right or moves to the right. */
  static int key_right;
  /** Scancode for grenade-key. */
  static int key_grenade;
  /** Scancode for Detonate grenades key */
  static int key_detonate;

  /** How many points the firstaid-pack heals. */
  static int firstaid_points;
  
  /** Dynamic wall shadows on/off. */
  static bool wallShadows;
  
  /** Lightsources on/off */
  static bool lightSources;
  
  /** Shooting lights on/off */
  static bool shootingLights;
  
  /** Explosions lights on/off */
  static bool explosionLights;

  /** Menupart musics on/off */
  static bool titleMusicOn;  
  /** ingame musics on or off. */
  static bool musicOn;
  /** ingame sounds on or off. */
  static bool soundsOn;
  
  /** Start from this level. */
  static int startFromLevel;

  /** The levelfile-list is here */
  static eng2d::FileList levelList;
  

  /** Datatype for difficulty settings. */
  //typedef int TDifficulty;
  
  /** Easy gamelevel */
  static const int KEasy = 0;
  
  /** Normal difficulty level */
  static const int KNormal = 1;
  
  /** Hard difficulty level */
  static const int KHard = 2;
  

  /** The current difficultylevel */
  static int difficultyLevel;
  
  /** Size of the lightbeam matrix cell. */
  static int lightbeamMatrixCellSize;

  ///
  /// Scores
  /// ======
  
  /** Number of score received when hitting an alien. */
  static int SCORE_BULLETHIT;
  /** Number of score received on each time player shoots */
  static int SCORE_SHOOT;
  /** Score granted when killing a CarnivoreAlien */
  static int SCORE_CARNIVOREALIEN;
  /** Scores granted when killing a SmallWormAlien */
  static int SCORE_SMALLWORMALIEN;
  /** Scores granted when killing a WingedAlien */
  static int SCORE_WINGEDALIEN;
  /** Scores granted when killing a ProctorAlien */
  static int SCORE_PROCTORALIEN;
  /** Scores granted when killing a MinigunAlien */
  static int SCORE_MINIGUNALIEN;
  /** Scores granted when killing a PredatorAlien */
  static int SCORE_PREDATORALIEN;



  ///
  /// Cheats
  /// ======
  
  /** Unlimited lifes - cheat on or off */
  static bool cheatLifes;
  /** Unlimited ammo - cheat on or off */
  static bool cheatAmmo;
  /** Immortal Youko - cheat on or off */
  static bool cheatYouko;
  
  
  static bool detonateDown;
  
  
  ///
  /// Settings for the GameObjects
  /// ============================
  
  /** The GameObject settings are stored in this vector. The
   * <code> init() </code> - method reads the configuration file and
   * places the found setups here.
   */
  static std::vector< Utils::GObjectConfig > gameObjectSettings;
  
  /** In this vector there are strings that describe the on-screen names
   * of the bonuses. The strings should be in order so that we can use
   * the @c BonusObject::Type - members as indices, such as
   * BonusObject::FIRSTAID
   */
  static std::vector< std::string > bonusObjectNames;
  
  /** Returns the requested GameObject property. These properties are loaded
   * from the configuration file. If the requested property does not exist,
   * this method throws an exception.
   * @param     objectID          The object id of the GameObject being returned
   * @param     propName          Name of the property being returned.
   * @return    A string that holds the value.
   * @exception Exception         Thrown if there is no requested property.
   */
  static std::string getObjectProperty( ObjectID::Type objectID,
                                 const std::string& propName ) throw ( Exception );

  /** Returns the requested GameObject property as <code>int</code>. If
   * there is no such property, an exception will be thrown.
   * @param     oid               The object id
   * @param     propName          Name of the property
   * @return    The requested property that is already converted to an integer.
   * @exception Exception         Thrown if there is no requested property.
   */
  static int intObjProp( ObjectID::Type oid, const std::string& propName )
                throw ( Exception );
                
  /** Returns the requested GameObject property as <code>float</code>. If
   * there is no such property, an exception will be thrown.
   * @param     oid               The object id
   * @param     propName          Name of the property
   * @return    The requested property as float.
   * @exception Exception         Thrown if there is no requested property.
   */
  static float floatObjProp( ObjectID::Type oid, const std::string& propName )
                  throw ( Exception );
                  
  /** Returns the string that contains the name of the given BonusObject
   * type.
   * @param     bonusID           The ID-code of the BonusObject
   * @return    A string that contains the name of the bonusitem.
   */
  static std::string getBonusObjectName( int bonusID );

  
  
  ///
  /// Settings for the Weapons
  /// ========================
  
  /** The Weapon settings are stored in this vector. The <code> init() </code> -
   * method reads the configuration file and pushes the found Weapon
   * settings here.
   */
  static std::vector< Utils::WeaponSpecsConfig > weaponSpecs;
  
  /** Returns Weapon properties. These properties are loaded from the
   * configuration file suring the startup process of the game. If
   * the requested property cannot be found, this method throws an
   * exception.
   * @param     wid               The weapon id such as Weapon::W_RIFLE
   * @param     propName          The name of the property
   * @return    A string that holds the value of the requested property.
   * @exception Exception         Thrown if the requested property cannot be
   *                              found.
   */
  static std::string getWProp( int wid, const std::string& propName )
                        throw ( Exception );
  
  /** Returns the requested parameter that is already converted to integer
   * @param     wid               The weapon id such as Weapon::W_SHOTGUN
   * @param     propName          The name of the parameter
   * @return    The requested parameter as an integer.
   * @exception Exception         Thrown if we were not able to find the
   *                              requested parameter
   */
  static int intWProp( int wid, const std::string& propName )
                    throw ( Exception );
                
                
  /** Returns the requested parameter that is already converted to float
   * @param     wid               The weapon id such as Weapon::W_SHOTGUN
   * @param     propName          The name of the parameter
   * @return    The requested parameter as an float.
   * @exception Exception         Thrown if we were not able to find the
   *                              requested parameter
   */
  static float floatWProp( int wid, const std::string& propName )
                        throw ( Exception );
                        
                        
  /** Constructor. No need to use.
   */
  Settings();
  /** Destructor.
   */
  ~Settings();
  

  /** Public methods **/

  /** Inits the settings. Loads them from configuration file.
   * @return    Nonzero if fails.
   */
  static int init();
  
  /** Saves the general settings, the settings from Settings-menu to
   * configuration file.
   */
  static void saveGeneralSettings();

  /** Public GET-methods **/

protected:

  /** Loads the main settings for the GameObject and Weapon elements.
   * @return    Nonzero if fails.
   */
  static int loadSettings();
  
  /** Reads the contents of SETTINGS - element.
   * @param     rFin              The input filestream
   * @return    0 on susccess, nonzero if fails.
   */
  static int readElement_Settings( std::ifstream& rFin );
  
  /** Reads the contents of SCORE - element.
   * @param     rIn               The input filestream
   * @return    0 on susccess, nonzero if fails.
   */
  static int readElement_Score( std::istream& rIn );
  
  /** Reads the contents of GAMEOBJECTS - element.
   * @param     rIn               The input stream
   * @return    0 on susccess, nonzero if fails.
   */
  static int readElement_GameObjects( std::istream& rIn );
  
  /** Reads the data of a single gameobject and pushes it to the
   * gameObjectSettings - vector.
   *
   * This is a "generic" method since it supports all the gameobjects.
   * @param     t                 Type of the object we're about to read.
   * @param     rFin              The input stream
   * @param     rEndTag           A string that tells the ending tag of this
   *                              element.
   * @return    Nonzero if fails.
   */
  static int readGameObject( ObjectID::Type t, std::istream& rFin,
                             const std::string& rEndTag );
                             
  /** Initializes the bonusobjects.
   */
  static int initBonusObjects();
                             
  /** Reads the data of a WEAPON-element
   * @param     rIn               The input stream
   * @return    Nonzero if fails
   */
  static int readElement_Weapons( std::istream& rIn );
  
  /** Reads the information of a single weapon and pushes it to the
   * weaponSpecsSettings - vector.
   * @param     w                 The type of this weapon.
   * @param     rIn               The input stream
   * @param     rEndTag           The ending tag
   * @return    Nonzero if fails.
   */
  static int readWeaponSpecs( int w, std::istream& rIn,
                              const std::string& rEndTag );
  
  
  

  /** Private methods **/

  Settings(const Settings& rO);
  Settings& operator = (const Settings& rO);

  /** Members **/

};


};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: settings.h,v $
 * Revision 1.3  2006/06/21 16:10:42  lamminsa
 * More light settings added.
 *
 * Revision 1.2  2006/06/17 21:46:03  lamminsa
 * lightbeamMatrixCellSize member added.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.1  2005-12-30 23:41:36+02  lamminsa
 * New flag: titleMusicOn
 *
 * Revision 1.0  2005-11-06 01:17:14+02  lamminsa
 * Initial revision
 *
 */
